
#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
FASTA ヘッダー名変換スクリプト（CDS と Proteins の 2 ファイル対応版）
- CDS 側ヘッダーから gene（GeneID）/ protein(product)/ protein_id を抽出
- 出力ヘッダー: >PROTID gene:GENEID transcript:PROTID description:Protein-Description
- PROTID の '_' は削除（例: XP_005793874.1 → XP005793874.1）、 '.' は保持
"""

from collections import OrderedDict
import re
from typing import Dict, Tuple, Optional

# ==============================
# 入出力ファイル設定（スクリプト先頭）
# ==============================
# --- 実データを使う場合はこちらを設定 ---
IN_CDS = "Emiliania-huxleyi-RS_cds_from_genomic.fna"
IN_AA  = "Emiliania-huxleyi-RS_protein.faa"
OUT_CDS = "out_Emiliania-huxleyi-RS_cds_from_genomic.fna"
OUT_AA  = "out_Emiliania-huxleyi-RS_protein.faa"

# --- 公開用サンプル、5レコード（ご提示の example_* ） ---
#IN_CDS  = "example_cds_5.fna"
#IN_AA   = "example_proteins_5.faa"
#OUT_CDS = "example_out_cds_5.fna"          # 例に合わせて命名
#OUT_AA  = "example_out_proteins_5.faa"     # 例に合わせて命名

def read_fasta_dict(filename: str) -> "OrderedDict[str, str]":
    """FASTA を順序付き dict に読み込み（ヘッダー行をキー、配列連結文字列を値）"""
    seq_dict = OrderedDict()
    with open(filename, "r") as infile:
        name = None
        for line in infile:
            line = line.strip()
            if not line:
                continue
            if line.startswith(">"):
                name = line
                seq_dict[name] = ""
            elif name:
                seq_dict[name] += line
            else:
                raise ValueError("ERROR: ヘッダーの前に配列があります。")
    return seq_dict

# ---------- ユーティリティ ----------
def sanitize_protid(pid: str) -> str:
    """PROTID の '_' を削除。'.' は保持。"""
    return pid.replace("_", "")

def extract_between_brackets(header: str) -> Dict[str, str]:
    """
    ヘッダー中の [ ... ] ブロックをすべて抽出し、key=value / key:val を辞書化。
    - [key=value] をそのまま登録
    - [db_xref=...,GeneID:17287457,...] 等の複合は、',' で分解し 'GeneID:17287457' を key:value として追加
    - [GeneID:xxx] のような 'key:val' 形式も登録
    """
    fields: Dict[str, str] = {}
    for m in re.finditer(r"\[([^\]]+)\]", header):
        content = m.group(1)
        if "=" in content:
            k, v = content.split("=", 1)
            fields[k.strip()] = v.strip()
            # さらに 'db_xref' 等の中を走査して 'GeneID:xxx' を拾う
            for part in content.split(","):
                part = part.strip()
                if ":" in part and "=" not in part:
                    kk, vv = part.split(":", 1)
                    # 既存登録よりも具体的なサブキーなら上書き
                    fields[kk.strip()] = vv.strip()
        else:
            # [GeneID:xxx] のような 'key:val'
            if ":" in content:
                kk, vv = content.split(":", 1)
                fields[kk.strip()] = vv.strip()
    return fields

def parse_cds_header(name_line: str) -> Tuple[Optional[str], Optional[str], Optional[str]]:
    """
    CDS ヘッダーから (geneID, protein_id, description) を抽出。
    geneID: [gene=...] / [GeneID:...] / [db_xref=...,GeneID:...] を優先順で探索
    description: [protein=...] が最優先、なければ [product=...]
    protein_id: [protein_id=...]
    """
    fields = extract_between_brackets(name_line)

    # geneID 探索優先順
    geneID = None
    for key in ("gene", "GeneID", "locus_tag"):
        if key in fields and fields[key]:
            geneID = fields[key]
            break
    if not geneID:
        # ヘッダー全体からも救済検索（db_xref に含まれるケースなど）
        m = re.search(r"GeneID[:=]\s*([A-Za-z0-9_.-]+)", name_line)
        if m:
            geneID = m.group(1)

    # description
    description = None
    for key in ("protein", "product"):
        if key in fields and fields[key]:
            description = fields[key]
            break

    # protein_id
    protID = fields.get("protein_id")

    return geneID, protID, description

# ==============================
# 1) CDS 側の処理：名称変換テーブル作成＋CDS 出力
# ==============================
seq_dict_DNA = read_fasta_dict(IN_CDS)
dic_name_change: Dict[str, str] = {}

with open(OUT_CDS, "w") as out_DNA:
    for name, seq in seq_dict_DNA.items():
        geneID, protID, description = parse_cds_header(name)

        if not protID:
            print("WARNING: protein_id を見つけられませんでした:", name)
            continue

        protID_s = sanitize_protid(protID)

        # gene / description の救済
        if not geneID:
            geneID = "NA"
            print("WARNING: geneID を見つけられませんでした。'NA' を使用:", name)
        if not description:
            description = "NA"
            print("WARNING: description を見つけられませんでした。'NA' を使用:", name)

        descriptionTMP = description.replace(" ", "-")
        newName = f">{protID_s} gene:{geneID} transcript:{protID_s} description:{descriptionTMP} "

        dic_name_change[protID_s] = newName
        out_DNA.write(newName + "\n")
        out_DNA.write(seq + "\n")

# ==============================
# 2) Proteins 側の処理：CDS 由来のヘッダーへ置換して出力
# ==============================
seq_dict_AA = read_fasta_dict(IN_AA)

with open(OUT_AA, "w") as out_AA:
    for name, seq in seq_dict_AA.items():
        # 先頭トークン（> の直後の非空白連続）を protein の ID とみなす
        m = re.search(r"^>(\S+)", name)
        if not m:
            print("WARNING: タンパク質側ヘッダーを解析できませんでした:", name)
            continue
        protID = sanitize_protid(m.group(1))

        if protID in dic_name_change:
            newName = dic_name_change[protID]
            out_AA.write(newName + "\n")
            out_AA.write(seq + "\n")
        else:
            # マッピングが見つからない場合も継続（警告のみ）
            print("WARNING: 対応する protein_id が CDS に見つかりません:", name)
            # 最低限のヘッダーで出力する（必要ならコメントアウト）
            out_AA.write(f">{protID} gene:NA transcript:{protID} description:NA \n")
            out_AA.write(seq + "\n")

if __name__ == "__main__":
    print("Done: ", OUT_CDS, " / ", OUT_AA)
